/*
 *  GCD Beispiele.c
 *  GCD Beispiele
 *
 *  Created by Kai Brüning on 18.10.09.
 *  Copyright 2009 Heise. All rights reserved.
 *
 */

#include <dispatch/dispatch.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include "Besipiele.h"
#include "Support.h"
#include "CallbackPrototypes.h"

/*
 Dieses Kommandozeilenwerkzeug exerziert die Beispiele aus dem c’t Artikel über Grand Central Dispatch.
 
 Die Beispiele finden sich in Beispiele.h und Beispiele.c. Diese Datei (GCD Beispiele.c) enthält den Treibercode
 für die Beispiele.
 
 Support.h und Support.m enthalten zwei Unterstützungsfunktionen. Während die Beispiele und der Treibercode in
 reinem C geschrieben sind, benutzen die Unterstützungsfunktionen zur Arbeitserleichterung Apples Objective-C
 Foundation. Zum Verständnis des restlichen Codes ist es jedoch nicht notwendig, die Implementation der
 Unterstützungsfunktionen zu verstehen.
 
 Wenn "GCD Beispiele" ohne Parameter aufgerufen wird, wird nur das "Hallo World"-Beispiel ausgeführt.
 
 Alle Parameter müssen Dateisystempfade sein, die auf Textdateien oder Verzeichnisse zeigen. Verzeichnisse werden
 rekursiv auf Textdateien durchsucht und dann wird die komplette Liste von Textdateien an die "Count Word"-Beispiele
 übergeben.
 */

int fileCount;
const char** filePaths;
int remainingFiles;

int main (int argc, const char* argv[]) {
	// Alle Parameter müssen Dateipfade sein.
	
	// Das einfachste Beispiel braucht keine Parameter.
	HelloWorld();
	
	// Alle weiteren Beispiele brauchen mindestens ein Argument, um etwas zum Zählen zu haben.
	if (argc <= 1) {
		// HelloWorld() macht seine Ausgabe im Hintergrund. Da kein Mechanismus da ist, um auf diese Ausgabe
		// zu warten, lasse ich den Hauptthread eine Sekunde schlafen.
		sleep(1);
		return 0;
	}
	
	// Sammele alle Textdateien auf, die als Parameter angegeben sind. Verzeichnisse werden rekursiv nach Textdateien
	// durchsucht. Das wieder frei geben der Pfade spare ich mir, da der Prozess eh endet.
	// Ich benutze hier der Einfachheit halber globale Variablen, um die Parameter in den anderen Funktionen zur
	// Verfügung zu haben.
	CollectTextFiles(argc - 1, &argv[1], &fileCount, &filePaths);
	
	// Auch CountWordsInBackground1() gibt sein Ergebnis im Hintergrund ohne Synchronisation mit dem Hauptthread aus.
	CountWordsInBackground1(filePaths[0]);

	// CountWordsInBackground2() ist das erste Beispiel, dass sein Ergebnis an den Hauptthread zurück gibt.
	// Das nutze ich hier (in IntegrateWordCountInGUI()) zur Ausgabe auf die Konsole und zum Starten des nächsten
	// Beispiels.
	CountWordsInBackground2(filePaths[0]);
	
	// Damit der Hauptthread Blöcke von der main queue verarbeiten kann, muss eine Schleife laufen, die die Blöcke
	// ausführt. In Cocoa-Programmen erledigt das die NSRunLoop, aber ein Kommandozeilenprogramm hat per Default
	// keine run loop. Dafür gibt es dispatch_main(), das in einer Schleife Blöcke von der main queue ausführt und
	// nie zurückkehrt.
	dispatch_main();
	
	return 0;	// wird nie erreicht
}


void IntegrateWordCountInGUI(int wordCount) {
	printf("The word count for the GUI is %i\n", wordCount);
	
	// Starte das nächsten Beispiel auf der main queue.
	dispatch_async(dispatch_get_main_queue(), ^{
		// Die globale Variable 'remainingFiles' verwende ich in IntegrateWordCountOfOneFileInGUI(), um das Ende
		// dieses Beispiels zu erkennen.
		remainingFiles = fileCount;
		CountWordsInBackground3(filePaths, fileCount);
	});
}

void IntegrateWordCountOfOneFileInGUI(const char* filePath, int wordCount) {
	printf("The word count of %s is %i\n", filePath, wordCount);
	
	// CountWordsInBackground3() ist fertig, wenn alle Dateien gezählt wurden.
	if (--remainingFiles == 0) {
		// Die letzten drei Beispiele sind synchron und können einfach eines nach dem anderen ausgeführt werden.
		dispatch_async(dispatch_get_main_queue(), ^{
			int totalCount;
			
			totalCount = CountWordsSync1(filePaths, fileCount);
			printf("CountWordsSync1 returns word count of %i\n", totalCount);
			
			totalCount = CountWordsSync2(filePaths, fileCount);
			printf("CountWordsSync2 returns word count of %i\n", totalCount);
			
			totalCount = CountWordsSync3(filePaths, fileCount);
			printf("CountWordsSync3 returns word count of %i\n", totalCount);
			
			// Da dispatch_main() nie zurückkehrt, muss das Programm mit dieser brutalen Methode beendet werden.
			exit (0);
		});
	}
}
